/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using DataAccess;
using MDWSLib.MDWSEmrSvc;
using System;
using System.Configuration;
using System.Collections.Generic;
using System.Data;
using System.Text;
using System.Text.RegularExpressions;
using System.Web;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.HtmlControls;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;

//
//reference: http://msdn.microsoft.com/en-us/library/c8y19k6h.aspx
//
//This is our base master page, all other master pages will derive from this
//master page. This page will handle db connection and cleanup etc...
//
//By defining the BaseMasterPage class as abstract, it can't be 
//created directly and can only serve as the base for another class.
//
//sequence of events for reference...
//Master page controls Init event
//Content controls Init event
//
//Master page Init event
//Content page Init event
//
//Content page Load event
//Master page Load event
//
//Content page PreRender event
//Master page PreRender event
//
//Master page controls PreRender event
//Content controls PreRender event
//
public abstract class BaseMaster : System.Web.UI.MasterPage
{
    private const string strLOGIN_PAGE = "TBICDS.aspx";
    private const string strPAGE_EXPIRED = "page_expired.aspx";
    private const string strCPRS_LOGIN = "TBICDSXPat.aspx";

    //we now defualt to the quick entry screen
    //private const string strPAT_SUMMARY = "tbipat_summary.aspx";
    private const string strPAT_SUMMARY = "pat_quickentry.aspx";

    /// <summary>
    /// set when we are accessed from the thin client
    /// or duz and dfn are passed on the query string
    /// </summary>
     public string DUZ
     {
        get
        {   string strValue = "";
             if(Session["DUZ"] != null)
            {
                strValue = Session["DUZ"].ToString();
            }
            
            return strValue;
        }
        set { Session["DUZ"] =  Convert.ToString(value); }
    }

    /// <summary>
    /// 508 theme property
    /// </summary>
     public bool Theme508
     {
         get
         {
             bool bValue = false;
             if (Session["THEME508"] != null)
             {
                 bValue = Convert.ToBoolean(Session["THEME508"].ToString());
             }

             return bValue;
         }
         set { Session["THEME508"] = Convert.ToBoolean(value); }
     }

     public bool ThemeDefault
     {
         get
         {
             bool bValue = false;
             if (Session["THEME508"] != null)
             {
                 bValue = Convert.ToBoolean(Session["ThemeDefault"].ToString());
             }

             return bValue;
         }
         set { Session["ThemeDefault"] = Convert.ToBoolean(value); }
     }

     public bool ThemeNone
     {
         get
         {
             bool bValue = false;
             if (Session["THEME508"] != null)
             {
                 bValue = Convert.ToBoolean(Session["ThemeNone"].ToString());
             }

             return bValue;
         }
         set { Session["ThemeNone"] = Convert.ToBoolean(value); }
     }

     /// <summary>
     /// selected CPA ID
     /// </summary>
     public string SelectedCPAID
     {
         get
         {
             string strValue = "";
             if (Session["SelectedCPAID"] != null)
             {
                 strValue = Session["SelectedCPAID"].ToString();
             }

             return strValue;
         }
         set { Session["SelectedCPAID"] = Convert.ToString(value); }
     }

     /// <summary>
     /// selected CPA text
     /// </summary>
     public string SelectedCPAText
     {
         get
         {
             string strValue = "";
             if (Session["SelectedCPAText"] != null)
             {
                 strValue = Session["SelectedCPAText"].ToString();
             }

             return strValue;
         }
         set { Session["SelectedCPAText"] = Convert.ToString(value); }
     }

    /// <summary>
    /// set when we are accessed from the thin client
    /// or duz and dfn are passed on the query string
    /// </summary>
    public string DFN
    {
         get
         {
             string strValue = "";
             if (Session["DFN"] != null)
             {
                 strValue = Session["DFN"].ToString();
             }

             return strValue;
         }
         set { Session["DFN"] = Convert.ToString(value); }
     }

    /// <summary>
    /// CPRS SRV param set when we are accessed from the query string
    /// </summary>
    public string SRV
    {
        get
        {
            string strValue = "";
            if (Session["SRV"] != null)
            {
                strValue = Session["SRV"].ToString();
            }

            return strValue;
        }
        set { Session["SRV"] = Convert.ToString(value); }
    }

    /// <summary>
    /// CPRS PORT param set when we are accessed from the query string
    /// </summary>
    public string PORT
    {
        get
        {
            string strValue = "";
            if (Session["PORT"] != null)
            {
                strValue = Session["PORT"].ToString();
            }

            return strValue;
        }
        set { Session["PORT"] = Convert.ToString(value); }
    }

    /// <summary>
    /// event status
    /// </summary>
    public long EventStatus
    {
        get
        {
            string strValue = "";
            if (Session["EVT_STATUS"] != null)
            {
                strValue = Session["EVT_STATUS"].ToString();
            }

            return CDataUtils2.ToLong(strValue);
        }
        set { Session["EVT_STATUS"] = Convert.ToString(value); }
    }
    
    /// <summary>
    /// region id
    /// </summary>
    public long RegionID
    {
        get
        {
            string strValue = "";
            if (Session["REGION_ID"] != null)
            {
                strValue = Session["REGION_ID"].ToString();
            }

            return CDataUtils2.ToLong(strValue);
        }
        set { Session["REGION_ID"] = Convert.ToString(value); }
    }

    /// <summary>
    /// site id
    /// </summary>
    public long SiteID
    {
        get
        {
            string strValue = "";
            if (Session["SITE_ID"] != null)
            {
                strValue = Session["SITE_ID"].ToString();
            }

            return CDataUtils2.ToLong(strValue);
        }
        set { Session["SITE_ID"] = Convert.ToString(value); }
    }

    /// <summary>
    /// US:5716 
    /// patient lookup by duz
    /// </summary>
    /// <param name="strDUZ"></param>
    /// <param name="strDFN"></param>
    public void LookupPatientByDUZ(string strDUZ, string strDFN)
    {
        //build lookup string
        string strURL = String.Empty;
        strURL += "TBICDSXPat.aspx?p1=";
        strURL += CDataUtils2.Sanitize(strDUZ);
        strURL += "&p2=";
        strURL += CDataUtils2.Sanitize(strDFN);
        strURL += "&p3=";
        strURL += CDataUtils2.Sanitize(SRV);
        strURL += "&p4=";
        strURL += CDataUtils2.Sanitize(PORT);
        strURL += "&p5=na";

        Response.Redirect(strURL, true);
        return;
    }

    /// <summary>
    /// US:5716 
    /// redirects to our home page
    /// </summary>
    public void GotoHomePage()
    {
        Response.Redirect("TBICDSHome.aspx", true);
    }

    /// <summary>
    /// redirects to our patient summary page
    /// </summary>
    public void GotoPatientSummaryPage()
    {
        Response.Redirect(strPAT_SUMMARY, true);
    }

    /// <summary>
    /// redirects to patient demographics page
    /// </summary>
    public void GotoPatientDemographicsPage()
    {
        Response.Redirect("pat_demographics.aspx", true);
    }

    /// <summary>
    /// US:5716 
    /// redirects to our home page
    /// </summary>
    public void EXTGotoHomePage()
    {
        Ext.Net.X.Redirect("TBICDSHome.aspx");
    }

    /// <summary>
    /// redirects to our patient summary page
    /// </summary>
    public void EXTGotoPatientSummaryPage()
    {
        Ext.Net.X.Redirect(strPAT_SUMMARY);
    }

    /// <summary>
    /// redirects to patient demographics page
    /// </summary>
    public void EXTGotoPatientDemographicsPage()
    {
        Ext.Net.X.Redirect("pat_demographics.aspx");
    }

    /// <summary>
    /// redirects to start page
    /// </summary>
    public void GotoLoginPage()
    {
        Response.Redirect(strLOGIN_PAGE, true);
    }

    /// <summary>
    /// redirects to start page
    /// </summary>
    public void EXTGotoLoginPage()
    {
        Ext.Net.X.Redirect(strLOGIN_PAGE);
    }

    /// <summary>
    /// MDWS SUPPORT: is support for MDWS turned on?
    /// </summary>
    public bool MDWSTransfer
    {
        get
        {
            bool bMDWS = false;
            string strMDWS = "";
            if (System.Configuration.ConfigurationManager.AppSettings["MDWSTransfer"] != null)
            {
                strMDWS = System.Configuration.ConfigurationManager.AppSettings["MDWSTransfer"].ToString();
                if (strMDWS == "1")
                {
                    bMDWS = true;
                }
            }

            return bMDWS;
        }
    }

    /// <summary>
    /// MDWS context
    /// </summary>
    public string MDWSEmrSvcContext
    {
        get
        {
            string strContext = string.Empty;
            if (System.Configuration.ConfigurationManager.AppSettings["MDWSEmrSvcContext"] != null)
            {
                strContext = System.Configuration.ConfigurationManager.AppSettings["MDWSEmrSvcContext"].ToString();
            }

            return strContext;
        }
    }


    /// <summary>
    /// loads and returns a CData instance from properties of BaseMaster
    /// </summary>
    /// <returns></returns>
    public CData GetDataObject()
    {
        CData data = new CData(DBConn,
                               ClientIP,
                               FXUserID,
                               RegionID,
                               SiteID,
                               DFN,
                               ASPSessionID,
                               Session,
                               true);

        return data;
    }

    //04/16/2012 - Security Updates///////////////////////////
    /// <summary>
    /// ASP .NET session id
    /// </summary>
    public string ASPSessionID
    {
        get
        {
            return Context.Session.SessionID;
        }
    }

    /// <summary>
    /// time out
    /// </summary>
    int m_nTimeOut = 15; //defaults to 15
    public int Timeout
    {
        set
        {
            m_nTimeOut = value;
        }
        get
        {
            return m_nTimeOut;
        }
    }
    public int TimeoutAdjust
    {
        get
        {
            return 16;
        }
    }

    /// <summary>
    /// get/set session
    /// </summary>
    //DB session id  member, this gets set when the user logs in
    public string DBSessionID
    {
        get
        {
            string strDBSessionID = "";
            if (Session["DBSessionID"] == null)
            {
                return strDBSessionID;
            }

            strDBSessionID = Session["DBSessionID"].ToString();
            return strDBSessionID;
        }
        set
        {
            Session["DBSessionID"] = value;
        }
    }
   
    //////////////////////////////////////////////////////////
    //data connection member
    private CDataConnection m_DBConnection;

    //app specific controller not part of framework...
    public AppMaster APPMaster;
    
    /// <summary>
    /// get the database connection
    /// </summary>
    public CDataConnection DBConn
    {
        get { return m_DBConnection; }
    }

    public string Key { get; private set; }

    /// <summary>
    /// get/set status comment info, it will append... this allows us to keep
    /// a running list of errors if needed
    /// </summary>
    private string m_strStatusComment;
    public string StatusComment
    {
        get { return m_strStatusComment; }
        set {
                if (m_strStatusComment != "")
                {
                    string strEnding = "";
                    if (m_strStatusComment.Length >= 6)
                    {
                        strEnding = m_strStatusComment.Substring(m_strStatusComment.Length - 6);
                    }
                    if (strEnding != "<br />")
                    {
                        Session["StatusComment"] += "<br />";
                        m_strStatusComment += "<br />";
                    }
                }
               
                Session["StatusComment"] += value;
                m_strStatusComment += value;


                //if the status contains an oracle error
                //then show a more user-friendly error instead
                //example: ORA-00942: table or view does not exist
                //search for "ORA-" 
                if (m_strStatusComment.ToUpper().IndexOf("ORA-") != -1)
                {
                    //todo: logg the current status in an error table

                    m_strStatusComment = "An error occured while processing, please contact your system administrator.";
                    m_strStatusComment += "<br />";
                    Session["StatusComment"] = m_strStatusComment;
                }
            }
    }

    /// <summary>
    /// clear the status, called by the masterpage after we display status info
    /// </summary>
    public void ClearStatus()
    {
        Session["StatusComment"] = "";
        Session["StatusCode"] = 0;

        m_strStatusComment = "";
        m_lStatusCode = 0;

    }

    /// <summary>
    /// get/set status code info
    /// </summary>
    private long m_lStatusCode;
    public long StatusCode
    {
        get { return m_lStatusCode; }
        set {
                //do not clear it if its already set to 1, this way
                //errors will overrule success for display
                if (m_lStatusCode < 1) //Changed because we have error status codes that are greater than 1.
                {
                    Session["StatusCode"] = value;
                    m_lStatusCode = value;
                }
            }
    }

    const string cnMasterSaveDiv = "divMasterSave";

    /// <summary>
    /// 
    /// </summary>
    /// <param name="bEnableButton"></param>
    /// <param name="strDirectMethod"></param>
    public void EnableMasterSaveButton(bool bEnableButton, string strDirectMethod) 
    {
        HtmlContainerControl divSaveBtn = (HtmlGenericControl)this.FindControl(cnMasterSaveDiv);
        if (divSaveBtn != null) {
            divSaveBtn.Visible = bEnableButton;
            if (bEnableButton) {
                if (!String.IsNullOrEmpty(strDirectMethod)) {
                    divSaveBtn.Attributes.Remove("onclick");
                    divSaveBtn.Attributes.Add("onclick", "if($('.x-form-invalid-field').length > 0) {App.lblISEMsg.setText(\"Please review the fields.\");App.winISE.show();}else{ " + strDirectMethod + " }");
                }
            }
        }
    }

    const string cnSystemFeedbackWindow = "winSysFeedback";
    const string cnSystemFeedbackPanel = "pnlSysFeedback";


    [Ext.Net.DirectMethod(IDMode = Ext.Net.DirectMethodProxyIDMode.None, ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    /// <summary>
    /// US:6069
    /// Show system feedbacks in an Ext.NET Window control
    /// </summary>
    public void ShowSystemFeedback(string strMessage, string strTitle) {
        
        //find the feedback Ext.NET window control
        Ext.Net.Window win = (Ext.Net.Window)this.FindControl(cnSystemFeedbackWindow);
        
        //sanitize the passed on string values
        string strFeedbackMsg = CDataUtils2.Sanitize(strMessage);
        string strFeedbackTitle = CDataUtils2.Sanitize(strTitle);

        //transform the message
        strFeedbackMsg = Regex.Replace(strFeedbackMsg, "~#\\(", "<", RegexOptions.IgnoreCase);
        strFeedbackMsg = Regex.Replace(strFeedbackMsg, "\\)#~", ">", RegexOptions.IgnoreCase);
        
        if (win != null) {

            //set the window title
            win.Title = strFeedbackTitle;

            //find the main contents panel
            foreach (Ext.Net.Panel pnlItem in win.Items) {
                if (pnlItem.ID == cnSystemFeedbackPanel) {
                    
                    //set the html content for the panel
                    pnlItem.Html = String.Format("<span>{0}</span>", strFeedbackMsg);
                }
            }

            //show the panel
            win.Show();
        }
    }

    //use this check to see if the user clicked the 
    //apps main "Save" button...
    public bool OnMasterSAVE()
    {
        //get the postback control
        string strPostBackControl = Request.Params["__EVENTTARGET"];
        if (strPostBackControl != null)
        {
            //did we do a patient lookup?
            if (strPostBackControl.IndexOf("btnMasterSave") > -1)
            {
                return true;
            }
        }

        return false;
    }
   
    /// <summary>
    /// get/set user id
    /// </summary>
    //user id member - set when we login
    private long m_lFXUserID;
    public long FXUserID
    {
        get {
                return m_lFXUserID; 
            }
        set { m_lFXUserID = value; }
    }

    //are we still logged in...
    public bool IsLoggedIn()
    {
        if (m_lFXUserID < 1)
        {
            return false;
        }

        return true;
    }

    /// <summary>
    /// get client ip, may be router but better than nothing
    /// </summary>
    public string ClientIP
    {
        get { return Context.Request.ServerVariables["REMOTE_ADDR"]; }
    }

    //sets a string viewstate value
    public void SetVSValue(string strKey, string strValue)
    {
        ViewState[strKey] = strValue;
    }

    //sets a bool viewstate value
    public void SetVSValue(string strKey, bool bValue)
    {
        ViewState[strKey] = bValue;
    }

    //sets a long viewstate value
    public void SetVSValue(string strKey, long lValue)
    {
        ViewState[strKey] = Convert.ToString(lValue);
    }

    //gets a string value from viewstate
    public string GetVSStringValue(string strKey)
    {
        string strValue = "";
        if (ViewState[strKey] != null)
        {
            strValue = Convert.ToString(ViewState[strKey]);
        }

        return strValue;
    }

    //gets a bool value from view state
    public bool GetVSBoolValue(string strKey)
    {
        bool bValue = false;
        if (ViewState[strKey] != null)
        {
            bValue = Convert.ToBoolean(ViewState[strKey]);
        }

        return bValue;
    }

    //gets a long value from viewstate
    public long GetVSLongValue(string strKey)
    {
        long lValue = -1;
        if (ViewState[strKey] != null)
        {
            lValue = Convert.ToInt32(ViewState[strKey]);
        }

        return lValue;
    }

    //closes the patient
    public void ClosePatient()
    {
        this.SelectedPatientID = "";
        this.LookupSearchCase = -1;

        //remove session variables associated to the patient
        Session["PATIENTNAME"] = null;
		Session["PAT_DEMOGRAPHICS_DS"] = null;
    }

    //this is the currently looked up patient id...
    public string SelectedPatientID
    {
        get
        {
            CSec sec = new CSec();
            string strValue = "";
            //more efficient to just use a session var
            //no db hit this way
            //GetSessionValue("SELECTED_PATIENT_ID", out strValue);
            if(Session["SELECTED_PATIENT_ID"] != null)
            {
                strValue = Session["SELECTED_PATIENT_ID"].ToString();
            }
            return sec.dec(strValue, "");
        }
        //set { SetSessionValue("SELECTED_PATIENT_ID", Convert.ToString(value)); }
        set 
        {
            CSec sec = new CSec();
            Session["SELECTED_PATIENT_ID"] =  sec.Enc(Convert.ToString(value), ""); 
        }
    }
    
    //this is the currently looked up provider id...
    public string SelectedProviderID
    {
        get
        {
            string strValue = "";
            //more efficient to just use a session var
            //no db hit this way
            //GetSessionValue("SELECTED_PROVIDER_ID", out strValue);
            if(Session["SELECTED_PROVIDER_ID"] != null)
            {
                strValue = Session["SELECTED_PROVIDER_ID"].ToString();
            }
            return strValue;
        }
        //set { SetSessionValue("SELECTED_PROVIDER_ID", Convert.ToString(value)); }
        set { Session["SELECTED_PROVIDER_ID"] =  Convert.ToString(value); }
   }

    // 03/11/2011 - this is the type of lookup:: 1- all cases, 2- open cases, 3- closed cases
    public long LookupSearchCase
    {
        get
        {
            string strValue = "-1";
            GetSessionValue("LOOKUP_SEARCH_CASE", out strValue);
            if (strValue.Length > 0)
            {
                return Convert.ToInt32(strValue);
            }
            else
            {
                return -1;
            }
        }
        set { SetSessionValue("LOOKUP_SEARCH_CASE", Convert.ToString(value)); }
    }
    
    /// <summary>
    /// constructor
    /// </summary>
    public BaseMaster()
    {
        //create a new dataconnection object
        m_DBConnection = new CDataConnection();

        //clear status
        m_strStatusComment = string.Empty;
        m_lStatusCode = -1;
        FXUserID = 0;
    }

    /// <summary>
    /// ping MDWS to keep it alive
    /// </summary>
    public void PingMDWS()
    {
        if (IsLoggedIn())
        {
            if (MDWSTransfer)
            {
                CMDWSOps ops = new CMDWSOps(this.GetDataObject());
                CStatus stat = new CStatus();
                stat = ops.IsMDWSValid();
                if (!stat.Status)
                {
                    //if we lose connection to MDWS tell the user so they
                    //can log off
                    this.ShowSystemFeedback(stat.StatusComment, "MDWS Connection Error");

                    //if we lose connection to MDWS go ahead and logoff...
                    //LogOff();
                }
            }
        }
    }

    /// <summary>
    /// this is the proper place to do initialization in a master page
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Init(object sender, EventArgs e)
    {
        //Every request is validate for authentication. 
        ValidateRequest();

        //app specific stuff outside the base controller
        APPMaster = new AppMaster();
        APPMaster.SetBaseMaster(this);

        //Returns a string that can be used in a client 
        //event to cause postback to the server. 
        Page.ClientScript.GetPostBackEventReference(this, String.Empty);

        //set the character set, since all pages derive from basemaster
        //this will set the encoding for all pages...
        Response.ContentEncoding = Encoding.UTF8;

        //init status info and objects
        m_strStatusComment = string.Empty;
        m_lStatusCode = -1;//-1 = success no show

        //04/16/2012 - Security Updates
        //set the timeout
        Timeout = (Session.Timeout < 15) ? Timeout = 15 : Timeout = Session.Timeout;

        //connect to the data source
        if (!ConnectToDataSource())
        {
            //redirect to an error page
            Response.Redirect("error_database.aspx");
        }

        //sec helper
        CSec sec = new CSec();
        
        //DBSessionID gets set in the database when the user
        //logs in. this is used to cache values in the db and to determine if the 
        //user is logged in
        //
        //reset FXUserID, only gets set in the call below
        FXUserID = 0;
        if (!String.IsNullOrEmpty(DBSessionID))
        {
            //get actual user id from the database session created when the 
            //user logs in
            string strUID = "";
            if (GetSessionValue("FX_USER_ID", out strUID))
            {
                if (strUID != "")
                {
                    FXUserID = Convert.ToInt32(strUID);
                }

                //load the app specific user details
                //needed for the application
                APPMaster.LoadUserDetails();
            }
            else
            {
                //log off if we cannot retrieve a valid session,
                //user timed out
                LogOff();
            }
        }

        //user does not have access to this page
        //so logoff.
        if (!sec.AuditPageAccess(this))
        {
            //redirect to an error page, if audits are failing then cant redirect to login page cause 
            //access cant be checked! so instead of logging off we redirect to the db error page
            KillSession();

            Response.Redirect("error_database.aspx");
        }

        string strPageName = this.GetPageName();
        if (strPageName.IndexOf("event_management.aspx") != -1)
        {
            Response.AddHeader("X-UA-Compatible", "IE=8,chrome=1");
        }
        else
        {
            Response.AddHeader("X-UA-Compatible", "IE=9,chrome=1");
        }
    }
      
    /// <summary>
    /// caches a session value in the database. 
    /// stored encrypted and more secure then an asp.net session var
    /// </summary>
    /// <param name="strKey"></param>
    /// <param name="strKeyValue"></param>
    /// <returns></returns>
    public bool SetSessionValue( string strKey,
                                 string strKeyValue)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        if (!IsLoggedIn())
        {
            return true;
        }

        //create a new parameter list
        CDataParameterList pList = new CDataParameterList();

        //add params for the DB call
        //
        //in paramaters
        //these will always be passed in to all sp calls
        pList.AddParameter("pi_vSessionID", DBSessionID, ParameterDirection.Input);
        pList.AddParameter("pi_vSessionClientIP", ClientIP, ParameterDirection.Input);
        pList.AddParameter("pi_nUserID", FXUserID, ParameterDirection.Input);
               
        //
        pList.AddParameter("pi_vKey", strKey, ParameterDirection.Input);
        pList.AddParameter("pi_vKeyValue", strKeyValue, ParameterDirection.Input);
        //
        //execute the stored procedure
        DBConn.ExecuteOracleSP("PCK_FX_SEC.SetSessionValue",
                                pList,
                                out lStatusCode,
                                out strStatusComment);
        
        // 0 = success if strStatus is populated it will show on the screen
       	// 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }
        
        return false;
    }

    /// <summary>
    /// gets a cached session value from the database. 
    /// stored encrypted and more secure then an asp.net session var
    /// </summary>
    /// <param name="strKey"></param>
    /// <param name="strKeyValue"></param>
    /// <returns></returns>
    public bool GetSessionValue(string strKey,
                                out string strKeyValue)
    {
        strKeyValue = "";

        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list
        CDataParameterList pList = new CDataParameterList();

        //in paramaters
        //these will always be passed in to all sp calls
        pList.AddParameter("pi_vDBSessionID", DBSessionID, ParameterDirection.Input);
        pList.AddParameter("pi_vWebSessionID", ASPSessionID, ParameterDirection.Input);
        pList.AddParameter("pi_vSessionClientIP", ClientIP, ParameterDirection.Input);
        pList.AddParameter("pi_nUserID", FXUserID, ParameterDirection.Input);

        //
        pList.AddParameter("pi_vKey", strKey, ParameterDirection.Input);
        pList.AddParameter("po_vKeyValue", strKeyValue, ParameterDirection.Output);
      
        //
        //execute the stored procedure
        DBConn.ExecuteOracleSP("PCK_FX_SEC.GetSessionValue",
                               pList,
                               out lStatusCode,
                               out strStatusComment);

        
        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            CDataParameter paramValue = pList.GetItemByName("po_vKeyValue");
            strKeyValue = paramValue.StringParameterValue;

            return true;
        }

        strKeyValue = "";
        return false;
    }

    /// <summary>
    /// deletes a cached session value in the database.
    /// </summary>
    /// <param name="strKey"></param>
    /// <returns></returns>
    public bool DeleteSessionValue(string strKey)
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list
        CDataParameterList pList = new CDataParameterList();

        //add params for the DB call
        //
        //in paramaters
        //these will always be passed in to all sp calls
        pList.AddParameter("pi_vSessionID", DBSessionID, ParameterDirection.Input);
        pList.AddParameter("pi_vSessionClientIP", ClientIP, ParameterDirection.Input);
        pList.AddParameter("pi_nUserID", FXUserID, ParameterDirection.Input);
       
        pList.AddParameter("pi_vKey", strKey, ParameterDirection.Input);
        //
        //execute the stored procedure
        DBConn.ExecuteOracleSP("PCK_FX_SEC.DeleteSessionValue",
                               pList,
                               out lStatusCode,
                               out strStatusComment);

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// deletes all cached session values in the database. 
    /// </summary>
    /// <returns></returns>
    public bool DeleteAllSessionValues()
    {
        //status info
        long lStatusCode = -1;
        string strStatusComment = "";

        //create a new parameter list
        CDataParameterList pList = new CDataParameterList();

        //add params for the DB call
        //
        //in paramaters
        //these will always be passed in to all sp calls
        pList.AddParameter("pi_vSessionID", DBSessionID, ParameterDirection.Input);
        pList.AddParameter("pi_vSessionClientIP", ClientIP, ParameterDirection.Input);
        pList.AddParameter("pi_nUserID", FXUserID, ParameterDirection.Input);
       
        //execute the stored procedure
        DBConn.ExecuteOracleSP("PCK_FX_SEC.DeleteAllSessionValues",
                               pList,
                               out lStatusCode,
                               out strStatusComment);

        // 0 = success if strStatus is populated it will show on the screen
        // 1 to n are errors and we always show errors
        if (lStatusCode == 0)
        {
            return true;
        }

        return false;
    }

    /// <summary>
    /// helper to get the current page name
    /// </summary>
    /// <returns></returns>
    public string GetPageName()
    {
        string strPath = System.Web.HttpContext.Current.Request.Url.AbsolutePath;
        System.IO.FileInfo oInfo = new System.IO.FileInfo(strPath);

        return oInfo.Name.ToLower();
    } 
   
    /// <summary>
    /// good place to close connections etc...
    /// </summary>
    public override void Dispose()
    {
        //close the database connection
        if (m_DBConnection != null) 
        {
            m_DBConnection.Close();
        }

        base.Dispose();
    }

    /// <summary>
    /// connect to datasource
    /// </summary>
    private bool ConnectToDataSource()
    {
        //get the connection string from the web.config file
        //connection string is encrypted in the file using MS recommended procedures
        //
        //cd\
        //cd windows
        //cd microsoft.net
        //cd framework
        //cd v2.0.50727
        //aspnet_regiis -pe "connectionStrings" -app "/PrimeCarePlus" -prov "RsaProtectedConfigurationProvider"
        //
        //look for connection strings in connection strings and app settings
        string strConnectionString = string.Empty;
        try
        {
            strConnectionString = ConfigurationManager.ConnectionStrings["DBConnString"].ConnectionString;
            Key = ConfigurationManager.ConnectionStrings["Key"].ConnectionString;
        }
        catch (Exception e)
        {
            string strStatus = e.Message;
        }

        bool bAudit = (ConfigurationManager.AppSettings["AUDIT"] == "1") ? true : false;

        //Connect to the database, connection is housed in the master page 
        //so that all pages that use the master have access to it.
        if (!m_DBConnection.Connect(
            strConnectionString,
            (int)DataConnectionType.Oracle,
            bAudit))
        {
            Session["DB_ERROR_CODE"] = string.Empty;
            Session["DB_ERROR"] = string.Empty;

            m_strStatusComment = "Error Connecting to Data Source";
            m_lStatusCode = 1;
            return false;
        }

        return true;
    }

    #region Page Request Control
    /// <summary>
    /// Every request is validate for authentication. 
    /// Validate the request if it's not a "Post Back" to check if we need to prompt for credentials.
    /// 
    /// After logging in the user can open more TABs without the need of reentering the credentials.
    /// But if the user closes all TABs or Logoff or the connection is not active (idle for more that the time permitted)
    /// the user most enter the credentials.
    /// </summary>
    private void ValidateRequest()
    {
        if (!Page.IsPostBack)
        {
            RegisterPageRequest();
            List<ActivePage> lstFXCrrPgs = (Session["FX_CRRPGS"] == null) ? new List<ActivePage>() : (List<ActivePage>)Session["FX_CRRPGS"];
            if (lstFXCrrPgs.FindAll(p => p.Status == "Active").Count == 1)
            {
                CheckRequestSource();
            }
        }

    }

    /// <summary>
    /// This method add the page name from the current request to the session list of current pages (session variable FX_CRRPGS)
    /// </summary>
    private void RegisterPageRequest()
    {
        List<ActivePage> lstFXCrrPgs = null; //Fx Current Pages
        string strPageName = String.Empty;

        //get the FX Current Pages list
        if (Session["FX_CRRPGS"] == null)
        {
            lstFXCrrPgs = new List<ActivePage>();
        }
        else
        {
            lstFXCrrPgs = (List<ActivePage>)Session["FX_CRRPGS"];
        }

        //get page name...
        strPageName = GetRequestPageName();

        if ("/" + strCPRS_LOGIN.ToLower() == strPageName.ToLower())
        {
            if (Session["FX_VIACPRS"] == null)
            {
                Session["FX_VIACPRS"] = 1;
            }
        }

        if ("/" + strCPRS_LOGIN.ToLower() != strPageName.ToLower())
        {
            foreach (ActivePage p in lstFXCrrPgs)
            {
                //expire any previous page...
                p.Status = "Expired";
            }


            //new active page...
            ActivePage ap = new ActivePage();
            ap.Name = strPageName;
            ap.Status = "Active";

            HtmlInputHidden htxtPageID = (HtmlInputHidden)this.FindControl("htxtPageID");
            if (htxtPageID != null)
            {
                htxtPageID.Value = ap.ID;
            }

            //add to list...
            lstFXCrrPgs.Add(ap);

            //set session var...
            Session["FX_CRRPGS"] = lstFXCrrPgs;

        }

    }

    /// <summary>
    /// Direct method to remove the page name from the lit of current pages (session variable FX_CRRPGS)
    /// This method is call from the MasterPage while navigating away from it (windows.onbeforeunload)
    /// </summary>
    [Ext.Net.DirectMethod(IDMode = Ext.Net.DirectMethodProxyIDMode.None, ShowMask = true, Msg = "<span role='alert'>Please wait.</span>")]
    public void UnregisterPageRequest()
    {
        List<ActivePage> lstFXCrrPgs = null; //Fx Current Pages
        string strPageName = String.Empty;

        //get the FX Current Pages list
        if (Session["FX_CRRPGS"] == null)
        {
            return;
        }
        else
        {
            lstFXCrrPgs = (List<ActivePage>)Session["FX_CRRPGS"];
        }

        //get page name...
        strPageName = GetRequestPageName();

        HtmlInputHidden htxtPageID = (HtmlInputHidden)this.FindControl("htxtPageID");
        if (htxtPageID == null)
        {
            //remove from list...
            lstFXCrrPgs.Remove(lstFXCrrPgs.Find(p => p.Name == strPageName));
        }
        else
        {
            //remove from list...
            lstFXCrrPgs.Remove(lstFXCrrPgs.Find(p => p.Name == strPageName && p.ID == htxtPageID.Value));
        }

        //set session var...
        Session["FX_CRRPGS"] = lstFXCrrPgs;

    }

    [Ext.Net.DirectMethod(IDMode = Ext.Net.DirectMethodProxyIDMode.None, ShowMask = false)]
    public object CheckPageExpired()
    {
        HtmlInputHidden htxtPageID = (HtmlInputHidden)this.FindControl("htxtPageID");
        if (htxtPageID != null)
        {
            List<ActivePage> lstFXCrrPgs = null; //Fx Current Pages

            //get the FX Current Pages list
            if (Session["FX_CRRPGS"] == null)
            {
                return new { StatusCode = 0, StatusComment = "" };
            }
            else
            {
                lstFXCrrPgs = (List<ActivePage>)Session["FX_CRRPGS"];
                ActivePage ap = lstFXCrrPgs.Find(p => p.ID == htxtPageID.Value);
                if(ap != null)
                {
                    if (ap.Status == "Expired")
                    {
                        return new { StatusCode = 1, StatusComment = "Page expired and will be closed." };
                    }
                }
            }
        }

        //if (SelectedPatientID != strPagePatientID)
        //{
        //    return new {StatusCode = 1, StatusComment = "Page expired and will be closed."};
        //    //X.Js.Call("pageExpired");
        //    //ShowSystemFeedback("Current patient change<br/>Need to close view...", "Closing tab...");
        //    //ScriptManager.RegisterStartupScript(this.Page, typeof(string), "showAlert", "Ext.onReady(function(){ var winSysFeedback = Ext.getCmp('winSysFeedback'); if(typeof(winSysFeedback) === 'object'){ winSysFeedback.show(); } });", true);
        //    //Page.ClientScript.RegisterOnSubmitStatement(typeof(Page), "closePage", "window.onunload = CloseWindow();");
        //}

        return new { StatusCode = 0, StatusComment = "" };
    }

    /// <summary>
    /// Returns the name of the page in the current request
    /// </summary>
    /// <returns></returns>
    private string GetRequestPageName()
    {
        string strPageName = String.Empty;

        //get page name...
        strPageName = Request.Url.ToString();

        
        //Clean
        //remove any query param...
        if (strPageName.IndexOf("?") > 0)
        {
            strPageName = strPageName.Substring(0, strPageName.IndexOf("?"));
        }
        //remove any slash at the beginnig or end...
        strPageName = strPageName.Trim('/');
        //remove anything prior to the page name...
        if (strPageName.IndexOf('/') > 0)
        {
            strPageName = strPageName.Substring(strPageName.LastIndexOf('/'));
        }

        return strPageName;
    }

    /// <summary>
    /// Check if the source of the request (Request.UrlReferrer) comes from inside the app.
    /// Gets information about the URL of the client's previous request that linked to the current URL.
    /// If this information is not available then the client could be trying to navigate to the app
    /// after closing the browser's tab (only scenario where SessionID still alive and the client could auto login 
    /// (if the session hasn't expired) if we don't validate this). 
    /// Another scenario could be that the client type the URL in the browser within the app. They will be logoff, if they do.
    /// 
    /// This works on:
    /// IE7 7.0.6002.18005 Cipher Strength 256-bit (latest security update)
    /// IE8 8.0.6001.18928 Cipher Strength 256-bit (latest security update)
    /// IE  9, 10, ...
    /// Google Chrome 41.0.2272.118 m
    /// </summary>
    private void CheckRequestSource()
    {
        //Check if we have a Session
        if (!String.IsNullOrEmpty(DBSessionID))
        {
            // Check if this request was referrer 
            //(we could add more validation here to check if the referrer is from our app)
            //We will Logoff the session if we don't have a referrer 
            // and they are navigating to any place other than the login page
            if (Request.UrlReferrer == null && Request.Url.ToString().ToLower().IndexOf(strLOGIN_PAGE.ToLower()) == -1)
            {
                long lVIACPRS = 0;
                if(Session["FX_VIACPRS"] != null)
                {
                    lVIACPRS = Convert.ToInt64(Session["FX_VIACPRS"]);
                }

                //get the request url
                string strReqURL = Request.Url.ToString().ToLower();

                //if cprs and (patient summary or login)
                if (lVIACPRS == 1 &&
                       (strReqURL.IndexOf(strPAT_SUMMARY.ToLower()) > -1
                         || strReqURL.IndexOf(strCPRS_LOGIN.ToLower()) > -1
                       )
                    )
                {
                    //pat summary
                    if (strReqURL.IndexOf(strPAT_SUMMARY.ToLower()) > -1)
                    {
                        //if navigating to TBIPAT_SUMMARY from CPRS link (TBICDSXPat) 
                        //we need to check how many pages are open. If just one then we 
                        //need to request credentials.
                        Session["FX_VIACPRS"] = null;
                        List<ActivePage> lstFXCrrPgs = (Session["FX_CRRPGS"] == null) ? 
                                                        new List<ActivePage>() : 
                                                        (List<ActivePage>)Session["FX_CRRPGS"];

                        if (lstFXCrrPgs.Count - lstFXCrrPgs.FindAll(p => p.Name == "/" + strLOGIN_PAGE).Count == 1)
                        {
                            //LoginViaCPRS();
                            LogOffCPRS();
                        }
                    }
                }
                else
                {
                    //hard logoff as the user should 
                    //not be opening pages in seperate tabs
                    LogOff();
                }
            }
        }
    }

    /// <summary>
    /// LogOff current user and redirect to login page as it will do a normal CPRS link.
    /// </summary>
    private void LogOffCPRS()
    {
        string strDUZ = String.Empty;
        string strDFN = String.Empty;
        string strSRV = String.Empty;
        string strPORT = String.Empty;

        //keeping the parameters to be able to simulate a normal
        //CPRS Login page otherwise it will ask for Region & Site
        strDUZ = DUZ;
        strDFN = DFN;
        strSRV = SRV;
        strPORT = PORT;

        //kill this session
        KillSession();
     
        //if we are not logged in set the duz, dfn etc...
        DUZ = strDUZ;
        DFN = strDFN;
        SRV = strSRV;
        PORT = strPORT;

        //login via cprs
        //build lookup string
        string strURL = String.Empty;
        strURL += "TBICDSXPat.aspx?p1=";
        strURL += CDataUtils2.Sanitize(strDUZ);
        strURL += "&p2=";
        strURL += CDataUtils2.Sanitize(strDFN);
        strURL += "&p3=";
        strURL += CDataUtils2.Sanitize(SRV);
        strURL += "&p4=";
        strURL += CDataUtils2.Sanitize(PORT);
        strURL += "&p5=na";

        //if invalid SRV then go to full login page
        if (String.IsNullOrEmpty(strSRV))
        {
            GotoLoginPage();
            return;
        }
        else
        {
            Response.Redirect(strURL, true);
            return;
        }
    }

    #endregion

    /// <summary>
    /// clears variables and kills the session
    /// </summary>
    public void KillSession()
    {
        //clear the patient
        this.ClosePatient();

        //clear FX_USER session var
        Session["FX_USER"] = null;

        //clear duz, dfn, srv and port
        Session["DUZ"] = null;
        Session["DFN"] = null;
        Session["SRV"] = null;
        Session["PORT"] = null;

        //clear Graphic Option session var
        Session["GRAPHIC_OPTION"] = null;

        //clear account details session var
        Session["ACC_DETAILS"] = null;

        //turn on ping
        Session["PING_MDWS"] = null;

        //do any clean up necessary to logoff
        CSec sec = new CSec();
        sec.LogOff(this);

        //is an extra step for timeouts etc...
        if (!String.IsNullOrEmpty(DBSessionID))
        {
            DeleteAllSessionValues();
        }

        //clear the dbsessionid
        DBSessionID = String.Empty;

        //clear the session
        Session.Clear();

        //abandon the session
        Session.Abandon();
    }

    /// <summary>
    /// US:5716
    /// called to logoff the user
    /// </summary>
    public void LogOff()
    {
        KillSession();
     
        //redirect;
        GotoLoginPage();
    }

    /// <summary>
    /// logoff called from EXT directmethods
    /// </summary>
    public void EXTLogOff()
    {
        //kill the session
        KillSession();

        //redirect;
        EXTGotoLoginPage();
    }
    

    /// <summary>
    /// setup logon from CPRS
    /// </summary>
    public void LoginViaCPRS()
    {
        //clear the patient
        this.ClosePatient();

        //clear FX_USER session var
        Session["FX_USER"] = null;
        Session["FX_CRRPGS"] = null;

        //Don't clear duz, dfn, srv and port
        //they are set from CPRS call
        //Session["DUZ"] = null;
        //Session["DFN"] = null;
        //Session["SRV"] = null;
        //Session["PORT"] = null;

        //clear Graphic Option session var
        Session["GRAPHIC_OPTION"] = null;

        //clear account details session var
        Session["ACC_DETAILS"] = null;

        //do any clean up necessary to logoff
        CSec sec = new CSec();
        sec.LogOff(this);

        //is an extra step for timeouts etc...
        if (!String.IsNullOrEmpty(DBSessionID))
        {
            DeleteAllSessionValues();
        }

        //clear the dbsessionid
        DBSessionID = String.Empty;

        //redirect;
        GotoLoginPage();
    }

    /// <summary>
    /// logoff
    /// </summary>
    /// <param name="bRedirect"></param>
    public void LogOff(bool bRedirect)
    {
        //kill the session
        KillSession();
        
        //redirect;
        if (bRedirect)
        {
            GotoLoginPage();
        }
    }

    /// <summary>
    /// active page
    /// </summary>
    private class ActivePage
    {
        public ActivePage()
        {
            this.ID = Guid.NewGuid().ToString();
        }

        public string Name { set; get; }
        public string ID { set;  get; }
        public string Status { set; get; }
    }
}